<?php
// This page shows all players registered for a specific tournament.
require_once 'header.php'; // Includes security check and db connection

// --- Validate Input and Fetch Tournament Title ---
$tournament_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if ($tournament_id === 0) {
    // A simple way to handle error, you could make a prettier error page
    echo "<p>Error: No tournament ID provided. Please go back and try again.</p>";
    require_once 'footer.php';
    exit();
}

// Fetch the tournament title to display on the page
$title_sql = "SELECT title FROM tournaments WHERE id = ?";
$stmt = $conn->prepare($title_sql);
$stmt->bind_param("i", $tournament_id);
$stmt->execute();
$tournament_title = $stmt->get_result()->fetch_assoc()['title'] ?? 'Unknown Tournament';
$stmt->close();

// --- Set the dynamic page title for the header ---
$page_title = "Registrations for: " . htmlspecialchars($tournament_title);
// This JavaScript updates the header H1 tag after the page has loaded the header
echo '<script>document.addEventListener("DOMContentLoaded", function() { document.querySelector(".top-header h1").textContent = "' . addslashes($page_title) . '"; });</script>';

// --- Fetch Registered Players ---
// We join the registrations and users tables to get the main account holder's details
$sql = "SELECT 
            u.username, u.email, u.mobile_number, 
            r.player_ign_data, r.registration_time, r.slots_taken
        FROM registrations r
        JOIN users u ON r.user_id = u.id
        WHERE r.tournament_id = ?
        ORDER BY r.registration_time ASC";

$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $tournament_id);
$stmt->execute();
$result = $stmt->get_result();

$registered_players = [];
if ($result && $result->num_rows > 0) {
    while($row = $result->fetch_assoc()) {
        $registered_players[] = $row;
    }
}
$conn->close();
?>
<style>
    /* Mobile-first responsive styles */
    .page-controls { 
        display: flex; 
        justify-content: flex-end; 
        margin-bottom: 20px; 
    }
    
    .btn-back { 
        background: var(--secondary); 
        color: white; 
        padding: 10px 20px; 
        border-radius: 8px; 
        text-decoration: none; 
        font-weight: 500; 
        display: inline-block;
        margin-bottom: 15px;
    }
    
    .data-container {
        width: 100%;
        background: var(--card-bg);
        border-radius: 12px;
        box-shadow: 0 4px 6px rgba(0,0,0,0.05);
        overflow: hidden;
    }
    
    /* Table styles for desktop */
    .data-table { 
        width: 100%; 
        border-collapse: collapse;
        display: table;
    }
    
    .data-table th, .data-table td { 
        padding: 15px; 
        text-align: left; 
        border-bottom: 1px solid #e5e7eb; 
    }
    
    .data-table th { 
        background: #f9fafb; 
        font-size: 12px; 
        color: var(--secondary); 
        text-transform: uppercase; 
        font-weight: 600; 
    }
    
    .data-table td { 
        font-size: 14px; 
    }
    
    .data-table tr:last-child td { 
        border-bottom: none; 
    }
    
    .ign-list { 
        list-style-type: none; 
        margin: 0; 
        padding: 0; 
    }
    
    .ign-list li { 
        margin-bottom: 5px; 
    }
    
    .ign-list li:last-child { 
        margin-bottom: 0; 
    }
    
    /* Mobile card view styles */
    .mobile-cards {
        display: none;
    }
    
    .player-card {
        padding: 15px;
        border-bottom: 1px solid #e5e7eb;
    }
    
    .player-card:last-child {
        border-bottom: none;
    }
    
    .card-row {
        display: flex;
        margin-bottom: 10px;
    }
    
    .card-label {
        font-weight: 600;
        min-width: 120px;
        color: var(--secondary);
        font-size: 12px;
        text-transform: uppercase;
    }
    
    .card-value {
        flex: 1;
    }
    
    /* Responsive adjustments */
    @media (max-width: 768px) {
        .data-table {
            display: none;
        }
        
        .mobile-cards {
            display: block;
        }
        
        .page-controls {
            justify-content: center;
        }
        
        .btn-back {
            width: 100%;
            text-align: center;
        }
    }
    
    @media (max-width: 480px) {
        .card-row {
            flex-direction: column;
            margin-bottom: 15px;
        }
        
        .card-label {
            margin-bottom: 5px;
        }
    }
</style>

<div class="page-controls">
    <a href="tournaments.php" class="btn-back">← Back to Tournaments List</a>
</div>

<!-- Desktop Table View -->
<div class="data-table">
    <table>
        <thead>
            <tr>
                <th>#</th>
                <th>Account Holder</th>
                <th>In-Game Names (Slots)</th>
                <th>Contact</th>
                <th>Registration Time</th>
            </tr>
        </thead>
        <tbody>
            <?php if (empty($registered_players)): ?>
                <tr>
                    <td colspan="5" style="text-align:center; padding: 30px;">No players have registered for this tournament yet.</td>
                </tr>
            <?php else: ?>
                <?php $count = 1; ?>
                <?php foreach ($registered_players as $player): ?>
                    <tr>
                        <td><?php echo $count++; ?></td>
                        <td><strong><?php echo htmlspecialchars($player['username']); ?></strong></td>
                        <td>
                            <?php 
                            // Decode the JSON data of player names
                            $ign_data = json_decode($player['player_ign_data'], true);
                            if (is_array($ign_data) && !empty($ign_data)) {
                                echo "<ul class='ign-list'>";
                                foreach($ign_data as $ign) {
                                    echo "<li>" . htmlspecialchars($ign) . "</li>";
                                }
                                echo "</ul>";
                                echo "<strong>(Slots: " . htmlspecialchars($player['slots_taken']) . ")</strong>";
                            } else {
                                echo "N/A";
                            }
                            ?>
                        </td>
                        <td><?php echo htmlspecialchars($player['mobile_number']); ?></td>
                        <td><?php echo date("d M Y, h:i A", strtotime($player['registration_time'])); ?></td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<!-- Mobile Card View -->
<div class="mobile-cards">
    <?php if (empty($registered_players)): ?>
        <div class="player-card" style="text-align:center; padding: 30px;">
            No players have registered for this tournament yet.
        </div>
    <?php else: ?>
        <?php $count = 1; ?>
        <?php foreach ($registered_players as $player): ?>
            <div class="player-card">
                <div class="card-row">
                    <div class="card-label">No.</div>
                    <div class="card-value"><?php echo $count++; ?></div>
                </div>
                
                <div class="card-row">
                    <div class="card-label">Account Holder</div>
                    <div class="card-value"><strong><?php echo htmlspecialchars($player['username']); ?></strong></div>
                </div>
                
                <div class="card-row">
                    <div class="card-label">In-Game Names</div>
                    <div class="card-value">
                        <?php 
                        $ign_data = json_decode($player['player_ign_data'], true);
                        if (is_array($ign_data) && !empty($ign_data)) {
                            echo "<ul class='ign-list'>";
                            foreach($ign_data as $ign) {
                                echo "<li>" . htmlspecialchars($ign) . "</li>";
                            }
                            echo "</ul>";
                            echo "<strong>(Slots: " . htmlspecialchars($player['slots_taken']) . ")</strong>";
                        } else {
                            echo "N/A";
                        }
                        ?>
                    </div>
                </div>
                
                <div class="card-row">
                    <div class="card-label">Contact</div>
                    <div class="card-value"><?php echo htmlspecialchars($player['mobile_number']); ?></div>
                </div>
                
                <div class="card-row">
                    <div class="card-label">Registered</div>
                    <div class="card-value"><?php echo date("d M Y, h:i A", strtotime($player['registration_time'])); ?></div>
                </div>
            </div>
        <?php endforeach; ?>
    <?php endif; ?>
</div>

<?php
require_once 'footer.php';
?>