<?php
/*
 * Tournament Add/Edit Form (with Match Type & Per Kill Prize)
 * /admin/tournament_form.php
 */
$page_title = "Tournament Form";
require_once 'header.php';

// --- Initialize Variables ---
$tournament = [
    'id' => null, 'title' => '', 'category' => 'br', 'match_type' => 'squad',
    'match_time' => '', 'entry_fee' => 0, 'prize_pool' => '', 'per_kill_prize' => 0,
    'max_players' => 48, 'image_url' => '', 'status' => 'upcoming', 'prize_details' => '[]',
    'room_id' => '', 'room_pass' => ''
];
$form_action = 'Create';
$error_message = '';

// --- Check if we are in EDIT mode ---
if (isset($_GET['id'])) {
    $form_action = 'Update';
    $tournament_id = (int)$_GET['id'];
    
    $sql = "SELECT t.*, rd.room_id, rd.room_pass 
            FROM tournaments t
            LEFT JOIN room_details rd ON t.id = rd.tournament_id
            WHERE t.id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $tournament_id);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($result->num_rows > 0) {
        $tournament = $result->fetch_assoc();
        if (is_null($tournament['prize_details'])) { $tournament['prize_details'] = '[]'; }
    } else { $error_message = "Tournament not found."; }
    $stmt->close();
}

// Decode prize details for the form
$prizes = json_decode($tournament['prize_details'], true);
if (json_last_error() !== JSON_ERROR_NONE || empty($prizes)) {
    $prizes = [['rank' => 1, 'prize' => 0]];
}

// --- Handle Form Submission ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Sanitize and retrieve main form data
    $id = isset($_POST['id']) ? (int)$_POST['id'] : null;
    $title = trim($_POST['title']);
    $category = trim($_POST['category']);
    $match_type = trim($_POST['match_type']);
    $match_time = trim($_POST['match_time']);
    $entry_fee = (int)$_POST['entry_fee'];
    $prize_pool = trim($_POST['prize_pool']);
    $per_kill_prize = (int)$_POST['per_kill_prize'];
    $max_players = (int)$_POST['max_players'];
    $image_url = trim($_POST['image_url']);
    
    
        // --- NEW: Default Image Logic ---


    // If the image_url submitted from the form is empty, set it to your default link.


    if (empty($image_url)) {


        $image_url = 'https://hunterbd71.kesug.com/images/logo.png';


    }
    
    $status = trim($_POST['status']);
    
    // Process Prize Details
    $prize_ranks = $_POST['prize_rank'] ?? [];
    $prize_values = $_POST['prize_value'] ?? [];
    $prize_details_array = [];
    for ($i = 0; $i < count($prize_ranks); $i++) {
        if (!empty($prize_ranks[$i]) && isset($prize_values[$i])) {
            $prize_details_array[] = ['rank' => (int)$prize_ranks[$i], 'prize' => (float)$prize_values[$i]];
        }
    }
    $prize_details_json = json_encode($prize_details_array);

    
    $room_id_input = trim($_POST['room_id']);
    $room_pass_input = trim($_POST['room_pass']);
    
    if (empty($title) || empty($category) || empty($match_time)) {
        $error_message = "Title, Category, and Match Time are required.";
    } else {
        $conn->begin_transaction();
        try {
            if ($id) { 
                $sql = "UPDATE tournaments SET title=?, category=?, match_type=?, match_time=?, entry_fee=?, prize_pool=?, per_kill_prize=?, max_players=?, image_url=?, status=?, prize_details=? WHERE id=?";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param("ssssisissssi", $title, $category, $match_type, $match_time, $entry_fee, $prize_pool, $per_kill_prize, $max_players, $image_url, $status, $prize_details_json, $id);
            } else { 
                $sql = "INSERT INTO tournaments (title, category, match_type, match_time, entry_fee, prize_pool, per_kill_prize, max_players, image_url, status, prize_details) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param("ssssisissss", $title, $category, $match_type, $match_time, $entry_fee, $prize_pool, $per_kill_prize, $max_players, $image_url, $status, $prize_details_json);
            }
            $stmt->execute();
            
            $tournament_id_for_room = $id ?? $conn->insert_id;
            if (!empty($room_id_input)) {
                $room_sql = "INSERT INTO room_details (tournament_id, room_id, room_pass) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE room_id = VALUES(room_id), room_pass = VALUES(room_pass)";
                $room_stmt = $conn->prepare($room_sql);
                $room_stmt->bind_param("iss", $tournament_id_for_room, $room_id_input, $room_pass_input);
                $room_stmt->execute();
            } else {
                $room_sql = "DELETE FROM room_details WHERE tournament_id = ?";
                $room_stmt = $conn->prepare($room_sql);
                $room_stmt->bind_param("i", $tournament_id_for_room);
                $room_stmt->execute();
            }
            
            $conn->commit();
            echo '<script>window.location.href = "tournament_form.php";</script>';

        } catch (Exception $e) {
            $conn->rollback();
            $error_message = "Database operation failed: " . $e->getMessage();
        }
    }
}
$conn->close();
?>
<style>
    .form-container { 
        max-width: 100%; 
        margin: 15px; 
        background: var(--card-bg); 
        padding: 20px; 
        border-radius: 12px; 
        box-shadow: 0 4px 6px rgba(0,0,0,0.05); 
    }
    
    .form-grid { 
        display: grid; 
        grid-template-columns: 1fr; 
        gap: 15px; 
    }
    
    .form-group { 
        display: flex; 
        flex-direction: column; 
        margin-bottom: 15px;
    }
    
    .form-group label { 
        font-weight: 500; 
        margin-bottom: 8px; 
        color: #374151; 
        font-size: 14px;
    }
    
    .form-group input, 
    .form-group select { 
        padding: 12px; 
        border: 1px solid #D1D5DB; 
        border-radius: 8px; 
        font-size: 16px; 
        width: 100%;
        box-sizing: border-box;
    }
    
    .form-actions { 
        margin-top: 25px; 
        display: flex; 
        justify-content: flex-end; 
    }
    
    .btn-submit { 
        background: var(--primary); 
        color: white; 
        padding: 14px 25px; 
        border: none; 
        border-radius: 8px; 
        font-weight: 600; 
        font-size: 16px; 
        cursor: pointer; 
        width: 100%;
    }
    
    .error-msg { 
        color: #DC2626; 
        background: #FEE2E2; 
        padding: 12px; 
        border-radius: 8px; 
        margin-bottom: 20px; 
        font-size: 14px;
    }
    
    .prize-section { 
        margin-top: 20px; 
        border-top: 1px solid #e5e7eb; 
        padding-top: 20px; 
    }
    
    .prize-entry { 
        display: flex; 
        flex-direction: column;
        gap: 10px; 
        margin-bottom: 15px; 
    }
    
    .prize-entry input { 
        width: 100%; 
    }
    
    .btn-remove-prize, 
    .btn-add-prize { 
        padding: 10px 15px; 
        font-size: 16px; 
        font-weight: 600; 
        border-radius: 8px; 
        cursor: pointer; 
        border: none; 
        width: 100%;
        margin-top: 5px;
    }
    
    .btn-remove-prize { 
        background: #FEE2E2; 
        color: #DC2626; 
    }
    
    .btn-add-prize { 
        background: #DBEAFE; 
        color: #2563EB; 
        margin-top: 10px;
    }
    
    /* Responsive adjustments */
    @media (min-width: 768px) {
        .form-grid {
            grid-template-columns: 1fr 1fr;
        }
        
        .prize-entry {
            flex-direction: row;
            align-items: center;
        }
        
        .btn-remove-prize,
        .btn-add-prize {
            width: auto;
        }
        
        .btn-submit {
            width: auto;
        }
    }
</style>

<div class="form-container">
    <h2><?php echo $form_action; ?> Tournament</h2>
    <?php if(!empty($error_message)): ?>
        <div class="error-msg"><?php echo htmlspecialchars($error_message); ?></div>
    <?php endif; ?>

    <form method="POST" action="tournament_form.php<?php if($tournament['id']) echo '?id=' . $tournament['id']; ?>">
        <?php if($tournament['id']): ?>
            <input type="hidden" name="id" value="<?php echo $tournament['id']; ?>">
        <?php endif; ?>

        <div class="form-group">
            <label for="title">Match Title</label>
            <input type="text" id="title" name="title" value="<?php echo htmlspecialchars($tournament['title']); ?>" required>
        </div>

                 <div class="form-group">


                <label for="category">Category</label>


                <select id="category" name="category">


                    <?php


                    // This is the array of all possible categories


                    $categories = ['br', 'cs', 'lone', 'cs2', 'ludo', 'free'];


                    


                    // --- THIS IS THE CORRECTED LOGIC ---


                    // 1. Determine the default category.


                    //    If a 'category' is passed in the URL (from the "Create New Match" button), use that.


                    //    Otherwise (if we are editing an existing match), use the category from the database.


                    $preselected_category = $_GET['category'] ?? $tournament['category'];





                    // 2. Loop through all possible categories


                    foreach ($categories as $cat) {


                        // 3. Compare the current category in the loop ($cat) with our pre-selected one.


                        //    If they match, add the 'selected' attribute.


                        $selected = ($preselected_category == $cat) ? 'selected' : '';


                        


                        // 4. Echo the final <option> tag.


                        echo "<option value='{$cat}' {$selected}>" . strtoupper($cat) . "</option>";


                    }


                    ?>


                </select>


            </div>
            
            <div class="form-group">
                <label for="match_type">Match Type</label>
                <select id="match_type" name="match_type">
                    <?php $types = ['solo', 'duo', 'squad']; 
                    foreach($types as $type){ 
                        $selected = ($tournament['match_type'] == $type) ? 'selected' : ''; 
                        echo "<option value='{$type}' {$selected}>" . ucfirst($type) . "</option>"; 
                    } ?>
                </select>
            </div>
            
           <div class="form-group">


    <label for="match_time">Match Time</label>


    <input type="datetime-local" id="match_time" name="match_time" value="<?php


        


        if (!empty($tournament['match_time'])) {


            // If we are EDITING and a time is already set, use that time.


            echo date('Y-m-d\TH:i', strtotime($tournament['match_time']));


        } else {


            // If we are CREATING, default to the current time in UTC+6 (Bangladesh).


            $date = new DateTime('now', new DateTimeZone('Asia/Dhaka'));


            echo $date->format('Y-m-d\TH:i');


        }


    ?>" required>


</div>
            
            <div class="form-group">
                <label for="entry_fee">Entry Fee</label>
                <input type="number" id="entry_fee" name="entry_fee" value="<?php echo htmlspecialchars($tournament['entry_fee']); ?>">
            </div>
            
            <div class="form-group">
                <label for="prize_pool">Total Prize Pool (Text)</label>
                <input type="text" id="prize_pool" name="prize_pool" value="<?php echo htmlspecialchars($tournament['prize_pool']); ?>" placeholder="e.g., 5000">
            </div>
            
            <div class="form-group">
                <label for="per_kill_prize">Per Kill Prize (Tk)</label>
                <input type="number" id="per_kill_prize" name="per_kill_prize" value="<?php echo htmlspecialchars($tournament['per_kill_prize']); ?>">
            </div>

            <div class="form-group">
                <label for="max_players">Max Players</label>
                <input type="number" id="max_players" name="max_players" value="<?php echo htmlspecialchars($tournament['max_players']); ?>">
            </div>
            
            <div class="form-group">
                <label for="status">Status</label>
                <select id="status" name="status">
                    <?php $statuses = ['upcoming', 'ongoing', 'completed', 'canceled']; 
                    foreach($statuses as $stat){ 
                        $selected = ($tournament['status'] == $stat) ? 'selected' : ''; 
                        echo "<option value='{$stat}' {$selected}>" . ucfirst($stat) . "</option>"; 
                    } ?>
                </select>
            </div>
        </div>

        <div class="form-group">
            <label for="image_url">Image URL (Optional)</label>
            <input type="text" id="image_url" name="image_url" value="<?php echo htmlspecialchars($tournament['image_url']); ?>" placeholder="https://example.com/image.jpg">
        </div>

        <div class="prize-section">
            <label>Prize Distribution</label>
            <div id="prize-list">
                <?php foreach($prizes as $prize): ?>
                    <div class="prize-entry">
                        <input type="number" name="prize_rank[]" placeholder="Rank (e.g., 1)" value="<?php echo htmlspecialchars($prize['rank']); ?>" required>
                        <input type="number" name="prize_value[]" placeholder="Prize Amount (e.g., 500)" value="<?php echo htmlspecialchars($prize['prize']); ?>" required>
                        <button type="button" class="btn-remove-prize" onclick="removePrize(this)">Remove</button>
                    </div>
                <?php endforeach; ?>
            </div>
            <button type="button" class="btn-add-prize" onclick="addPrize()">+ Add Prize</button>
        </div>

        <div class="prize-section">
            <label>Room Details</label>
            <p style="font-size: 12px; color: #666; margin-top: -10px; margin-bottom: 15px;"></p>
            <div class="form-grid">
                <div class="form-group">
                    <label for="room_id">Room ID</label>
                    <input type="text" id="room_id" name="room_id" value="<?php echo htmlspecialchars($tournament['room_id'] ?? ''); ?>">
                </div>
                <div class="form-group">
                    <label for="room_pass">Room Password</label>
                    <input type="text" id="room_pass" name="room_pass" value="<?php echo htmlspecialchars($tournament['room_pass'] ?? ''); ?>">
                </div>
            </div>
        </div>

        <div class="form-actions">
            <button type="submit" class="btn-submit"><?php echo $form_action; ?> Tournament</button>
        </div>
    </form>
</div>

<script>
    function addPrize() { 
        const list = document.getElementById('prize-list'); 
        const entry = document.createElement('div'); 
        entry.className = 'prize-entry'; 
        entry.innerHTML = `
            <input type="number" name="prize_rank[]" placeholder="Rank" required>
            <input type="number" name="prize_value[]" placeholder="Prize Amount" required>
            <button type="button" class="btn-remove-prize" onclick="removePrize(this)">Remove</button>
        `; 
        list.appendChild(entry); 
    }
    
    function removePrize(button) { 
        button.parentElement.remove(); 
    }
</script>

<?php
require_once 'footer.php';
?>