<?php
/*
 * Deposit Action Handler
 * /admin/deposit_action.php
 */
session_start();

// Security Check: Ensure admin is logged in
if (!isset($_SESSION['admin_id'])) {
    header("Location: login.php");
    exit();
}

// Check if it's a POST request with the required data
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['id'], $_POST['action'])) {
    
    require_once '../db_connect.php';
    
    $transaction_id = (int)$_POST['id'];
    $action = $_POST['action']; // 'approve' or 'decline'

    // Use a database transaction for this critical operation
    $conn->begin_transaction();

    try {
        // First, get the transaction details and lock the row to prevent race conditions
        $sql = "SELECT user_id, amount FROM transactions WHERE id = ? AND status = 'pending' AND type = 'deposit' FOR UPDATE";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("i", $transaction_id);
        $stmt->execute();
        $transaction = $stmt->get_result()->fetch_assoc();

        if ($transaction) {
            if ($action === 'approve') {
                // 1. Update the user's balance
                $update_balance_sql = "UPDATE users SET balance = balance + ? WHERE id = ?";
                $stmt = $conn->prepare($update_balance_sql);
                $stmt->bind_param("di", $transaction['amount'], $transaction['user_id']);
                $stmt->execute();

                // 2. Update the transaction status to 'completed'
                $update_txn_sql = "UPDATE transactions SET status = 'completed' WHERE id = ?";
                $stmt = $conn->prepare($update_txn_sql);
                $stmt->bind_param("i", $transaction_id);
                $stmt->execute();

            } elseif ($action === 'decline') {
                // Just update the transaction status to 'failed'
                $update_txn_sql = "UPDATE transactions SET status = 'failed' WHERE id = ?";
                $stmt = $conn->prepare($update_txn_sql);
                $stmt->bind_param("i", $transaction_id);
                $stmt->execute();
            }
            
            // If all steps succeeded, commit the changes
            $conn->commit();
            header("Location: deposits.php?status=success");
            exit();

        } else {
            // Transaction not found or already processed, roll back
            throw new Exception("Transaction not found or already processed.");
        }
    } catch (Exception $e) {
        // If any part fails, roll back the entire transaction
        $conn->rollback();
        // You can log the error $e->getMessage() for debugging
        die("An error occurred. The transaction has been canceled. Please go back and try again.");
    }

    $stmt->close();
    $conn->close();

} else {
    // Redirect if accessed directly or without proper data
    header("Location: deposits.php");
    exit();
}
?>